Webr.component.RichTextEditor = function (iframe) {
  this.fireEvents = ["click", "mousedown", "mouseup", "focus", "blur", "select", "change"];
  this.focused = false;
  this.iframe = $(iframe);
  this.initIframe();
  this.setValue(this.iframe.attr("_valueId"));
  this.setDefaultCursorPosition();
  $(iframe).data(Webr.component.ComponentRegistry.COMPONENT_INSTANCE, this);
};
Webr.component.RichTextEditor.prototype.initIframe = function () {
  var stylesheet = '${getClasspathUri("jetbrains/mps/webr/htmlComponent/css/richTextArea.css")}';
  this.idoc = this.iframe.get(0).contentWindow.document;
  $(this.idoc).attr("designMode", "on");
  var html = ["<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.0 Transitional//EN\" \"http://www.w3.org/TR/html4/loose.dtd\"><html><head>"];
  //ie8 specific work around
  html.push("<meta http-equiv=\"X-UA-Compatible\" content=\"IE=EmulateIE7\"/>");
  html.push("<link rel=\"stylesheet\" media=\"screen\" type=\"text/css\" href=\"" + stylesheet + "\"/>");
  html.push("</head><body class=\"" + "jt-richText" + "\"></body></html>");
  this.idoc.open();
  this.idoc.write(html.join(""));
  this.idoc.close();
  this.container = $(this.idoc.body);
  if (!Webr.util.Util.isIE) {
    this.container.attr("contentEditable", "true");
  }

  this.registerEvents();
};
Webr.component.RichTextEditor.prototype.registerEvents = function () {
  var it = this;
  if (Webr.util.Util.isOpera) {
    //cancel focus event from iframe window and leave from iframe body
    this.iframe.focus(function () {
      it.focused = true;
      info("RTE focused");
    });
    this.iframe.blur(function () {
      it.focused = false;
      info("RTE blurred");
    });
  }

  for (var i = 0; i < this.fireEvents.length; ++i) {
    this.regEvent(this.fireEvents[i]);
  }

};
Webr.component.RichTextEditor.prototype.regEvent = function (evnt) {
  var win = Webr.util.Util.isOpera ?$(this.iframe.get(0).contentWindow) :$(this.idoc);
  var iframe = this.iframe;
  var it = this;
  win.bind(evnt, function () {
    if (evnt == "focus" && it.focused) {
      return ;
    }

    iframe.trigger(evnt);
  });
};
Webr.component.RichTextEditor.prototype.highlight = function (value, rangesToHighlight) {
  if (value == this.getValue()) {
    var s = this.getTextSelection();
    var cur = 0;
    this.clear();
    var newContainer = this.container;
    for (var i = 0; i < rangesToHighlight.length; ++i) {
      var r = rangesToHighlight[i];
      if (cur < r.start) {
        this._appendText(newContainer, value.substring(cur, r.start));
      }

      if (r.start < r.end) {
        var span = $(this.idoc.createElement("span"));
        this._appendText(span, value.substring(r.start, r.end));
        span.addClass(r.styleClass);
        if (r.title) {
          span.attr("title", r.title);
        }

        newContainer.append(span);
      }

      cur = r.end;
    }

    if (cur < value.length) {
      this._appendText(newContainer, value.substring(cur, value.length));
    }

    this.setTextSelection(s);
  }

};
Webr.component.RichTextEditor.prototype._appendText = function (el, value) {
  var lines = value.split("\n");
  el.append(this.idoc.createTextNode(lines[0]));
  for (var i = 1; i < lines.length; ++i) {
    el.append(this.idoc.createElement("br"));
    el.append(this.idoc.createTextNode(lines[i]));
  }

};
Webr.component.RichTextEditor.prototype.getIECursorPosition = function (sel) {
  var body = this.container.text();
  var sPos;
  var ePos;
  ePos = sPos = body.length;
  var range = sel.createRange();
  var isCollapsed = range.compareEndPoints("StartToEnd", range) == 0;
  var id = this.iframe.attr("id") + "_cursor";
  if (isCollapsed) {
    var counter = 0;
    var start = "<b id=\"" + id + "\">";
    var end = "</b>";
    var cursor = start + "^" + end;
    range.pasteHTML(cursor);
    range.moveEnd("character", -end.length);
    range.collapse(false);
    range.select();
    while (range.expand("character")) {
      ++counter;
    }

    if (counter != 0) {
      var rPart = range.text.split("^")[1];
      if (rPart) {
        sPos = ePos = body.indexOf(rPart);
      }

    }

    cursor = this.idoc.getElementById(id);
    cursor.parentNode.removeChild(cursor);
    sel.empty();
  } else {
    var rText = range.text;
    sPos = body.indexOf(rText);
    ePos = sPos + rText.length;
  }

  return {start: sPos, end: ePos};
};
Webr.component.RichTextEditor.prototype.getTextSelection = function () {
  var root = this.container.get(0);
  if (root) {
    var sel = this.getWindowSelection();
    var selectionStart = {node: null, offset: null};
    var selectionEnd = {node: null, offset: null};
    var bookmark = null;
    if (Webr.util.Util.isIE) {
      var range = sel.createRange();
      var POS = this.getIECursorPosition(sel);
      selectionStart.offset = POS.start;
      selectionEnd.offset = POS.end;
      if (selectionStart.offset != selectionEnd.offset) {
        bookmark = range.getBookmark();
      }

      //non IE
    } else {
      selectionStart = {node: sel.anchorNode, offset: sel.anchorOffset};
      selectionEnd = {node: sel.focusNode, offset: sel.focusOffset};
    }

    return {start: selectionStart.node != null ?Webr.component.DomPositionData.toTextOffset(root, selectionStart.node, selectionStart.offset) :selectionStart.offset, end: selectionEnd.node != null ?Webr.component.DomPositionData.toTextOffset(root, selectionEnd.node, selectionEnd.offset) :selectionEnd.offset, bookmark: bookmark};
  }

};
Webr.component.RichTextEditor.prototype.setTextSelection = function (textSelection) {
  var root = this.container.get(0);
  if (root) {
    var sel = this.getWindowSelection();
    var range = this.createRangeFromSelection(sel);
    if (Webr.util.Util.isIE) {
      if (textSelection.bookmark == null) {
        range.moveToElementText(root);
        range.moveStart("character", textSelection.start);
        range.moveEnd("character", textSelection.end);
        if (textSelection.start == textSelection.end) {
          range.collapse(true);
        }

      } else {
        range.moveToBookmark(textSelection.bookmark);
      }

      range.select();
    } else {
      var textStart = textSelection.start < textSelection.end ?textSelection.start :textSelection.end;
      var textEnd = textSelection.start < textSelection.end ?textSelection.end :textSelection.start;
      var start = Webr.component.DomPositionData.toDomPosition(root, textStart);
      var end = Webr.component.DomPositionData.toDomPosition(root, textEnd);
            

      range.setStart(start.node, start.offset);
      range.setEnd(end.node, end.offset);
      sel.removeAllRanges();
      sel.addRange(range);
    }

    this.setFocus();
  }

};
Webr.component.RichTextEditor.prototype.getWindowSelection = function () {
  this.setFocus();
  return Webr.util.Util.isIE ?this.idoc.selection :this.iframe.get(0).contentWindow.getSelection();
};
Webr.component.RichTextEditor.prototype.createRangeFromSelection = function (sel) {
  if (sel) {
    if (Webr.util.Util.isIE) {
      return sel.createRange();
    } else {
      return this.idoc.createRange();
    }

  }

};
Webr.component.RichTextEditor.prototype.getCaretPosition = function () {
  return this.getTextSelection().end;
};
Webr.component.RichTextEditor.prototype.setDefaultCursorPosition = function () {
  if (this.container) {
    var sel = this.getWindowSelection();
    var range = this.createRangeFromSelection(sel);
    var element = this.container.get(0);
    if (Webr.util.Util.isIE) {
      range.moveToElementText(element);
      range.collapse(false);
      range.select();
    } else {
      range.selectNodeContents(element);
      range.collapse(false);
      sel.removeAllRanges();
      sel.addRange(range);
    }

    this.setFocus();
  }

};
Webr.component.RichTextEditor.prototype.getValue = function () {
  var root = this.container.get(0);
  return root ?Webr.component.DomPositionData.getTextValue(root) :"";
};
Webr.component.RichTextEditor.prototype.setValue = function (value) {
  var root = this.container;
  if (root && value) {
    this._appendText(root, value);
  }

};
Webr.component.RichTextEditor.prototype.clear = function () {
  this.container.empty();
};
Webr.component.RichTextEditor.prototype.oncaretmove = function (caretPos, value) {
};
Webr.component.RichTextEditor.prototype.fireCaretMove = function (caretPos, value) {
  $(this).trigger("caretmove", {caretPos: caretPos, value: value});
};
Webr.component.RichTextEditor.prototype.onvaluechange = function (caretPos, value) {
};
Webr.component.RichTextEditor.prototype.fireValueChange = function (caretPos, value) {
  $(this).trigger("valuechange", {caretPos: caretPos, value: value});
};
Webr.component.RichTextEditor.prototype.setFocus = function () {
  //It seems that Opera has two context for the focus: 1. window 2. iframe.contentWindow
  //call blur we guarantee a focus event invocation for the iframe window
  var element = Webr.util.Util.isOpera ?this.container :this.iframe.get(0).contentWindow;
  if (Webr.util.Util.isOpera) {
    element.blur();
  }

  element.focus();
};
Webr.component.RichTextEditor.prototype.setBlur = function () {
  var element = Webr.util.Util.isOpera ?this.container :this.iframe.get(0).contentWindow;
  element.blur();
};
Webr.component.RichTextEditor.registerRichTextEditor = function (path, suffixName) {
  cr.forEach(path, suffixName, function () {
    var rte = new Webr.component.RichTextEditor(this);
  });
};
Webr.component.TextSelectionData = function () {
};
Webr.component.DomPositionData = function () {
};
Webr.component.DomPositionData.toTextOffset = function (root, node, offset) {
  var textOffset = offset;
  for (var currentNode = node; currentNode != root && currentNode != null; currentNode = currentNode.parentNode) {
    for (var leftNeigboor = currentNode.previousSibling; leftNeigboor != null; leftNeigboor = leftNeigboor.previousSibling) {
      textOffset += Webr.component.DomPositionData.getTextLength(leftNeigboor);
    }

  }

  return textOffset;
};
Webr.component.DomPositionData.toDomPosition = function (root, textOffset) {
  var children = root.childNodes;
  for (var i = 0; i < children.length; ++i) {
    var child = children.item(i);
    if (child.nodeType == 3) {
      var len = child.nodeValue.length;
      if (len < textOffset) {
        textOffset -= len;
      } else {
        return {node: child, offset: textOffset};
      }

    } else if (child.nodeType == 1) {
      var el = child;
      var elTagName = el.tagName.toLowerCase();
      if (elTagName == "br") {
        if (1 < textOffset) {
          textOffset -= 1;
        } else {
          return {node: child, offset: textOffset};
        }

      } else if (elTagName == "div" || elTagName == "p") {
        if (el.childNodes.length == 1 && el.childNodes.item(0).nodeType == 1 && el.childNodes.item(0).tagName.toLowerCase() == "br") {
          if (1 < textOffset) {
            textOffset -= 1;
          } else {
            return {node: child, offset: textOffset};
          }

        } else {
          if (1 < textOffset) {
            textOffset -= 1;
          } else {
            return {node: child, offset: textOffset};
          }

          var len = Webr.component.DomPositionData.getTextLength(el);
          if (len < textOffset) {
            textOffset -= len;
          } else {
            return Webr.component.DomPositionData.toDomPosition(el, textOffset);
          }

        }

      } else {
        var len = Webr.component.DomPositionData.getTextLength(el);
        if (len < textOffset) {
          textOffset -= len;
        } else {
          return Webr.component.DomPositionData.toDomPosition(el, textOffset);
        }

      }

    }

  }

  return {node: root, offset: textOffset};
};
Webr.component.DomPositionData.getTextLength = function (node) {
  if (node.nodeType == 3) {
    return node.nodeValue.length;
  } else if (node.nodeType == 1) {
    var el = node;
    if (el.tagName.toLowerCase() == "br") {
      return 1;
    } else {
      var length = 0;
      var children = el.childNodes;
      for (var i = 0; i < children.length; ++i) {
        length += Webr.component.DomPositionData.getTextLength(children.item(i));
      }

      return length;
    }

  }

  return 0;
};
Webr.component.DomPositionData.getTextValue = function (node) {
  if (node.nodeType == 3) {
    return node.nodeValue;
  } else if (node.nodeType == 1) {
    var el = node;
    var elTagName = el.tagName.toLowerCase();
    if (elTagName == "br") {
      return "\n";
    } else if (elTagName == "div" || elTagName == "p") {
      if (el.childNodes.length == 1 && el.childNodes.item(0).nodeType == 1 && el.childNodes.item(0).tagName.toLowerCase() == "br") {
        return "\n";
      } else {
        //opera surrounds the 1st element with <p> tag
        var value = el.previousSibling == null ?"" :"\n";
        var children = el.childNodes;
        for (var i = 0; i < children.length; ++i) {
          value += Webr.component.DomPositionData.getTextValue(children.item(i));
        }

        return value;
      }

    } else {
      var value = "";
      var children = el.childNodes;
      for (var i = 0; i < children.length; ++i) {
        value += Webr.component.DomPositionData.getTextValue(children.item(i));
      }

      return value;
    }

  }

  return "";
};
var regrte = Webr.component.RichTextEditor.registerRichTextEditor;
