/*
 Copyright 2017 JetBrains s.r.o.

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 */

var entities = require('@jetbrains/youtrack-scripting-api/entities');
var jira = require('./jira');
var security = require('./security');
var dateTime = require('@jetbrains/youtrack-scripting-api/date-time');
var importUtils = require('@jetbrains/youtrack-scripting-api-import/utils');
var entityConverters = require('./entityConverters');
var events = require('./events');
var perTypeConverters = require('./perTypeConverters');
var importContext = require('./importContext');
var constants = require('./constants');

var extractAuthorsAndVisibilityGroups = function (jiraEntityName, valueGetter) {
  // TODO: get rid of valueGetter after synthetic jiraIssue implementation
  return function (ytFieldName, project, jiraProject, jiraFieldPrototype, jiraProjectField, jiraIssue) {
    if (jiraProjectField.allowedValues && jiraProjectField.allowedValues.length > 0) {
      valueGetter(jiraProjectField.allowedValues, jiraIssue).forEach(function (jiraEntity) {
        var jiraEntityInfo = jiraEntityName + ' ' + jiraEntity.id;
        if (importContext.timeInterval.includes(jiraProject.key, jiraEntity.updated ? jiraEntity.updated : jiraEntity.created)) {
          console.trace('Extracting authors and groups from ' + jiraEntityInfo);
          // null author will be converted to deleted user
          entityConverters.user(jiraEntity.author);
          if (jiraEntity.updateAuthor && (jiraEntity.updated !== jiraEntity.created)) {
            entityConverters.user(jiraEntity.updateAuthor);
          }
          jiraEntity.visibility && security.visibilityGroup(jiraEntity.visibility, jiraProject, project);
        } else {
          console.trace('Skipping authors and groups extracting from ' + jiraEntityInfo + ' - out of time interval');
        }
      });
    }
  };
};

var extractCommentsVisibility = extractAuthorsAndVisibilityGroups('comment', function (allowedValues, jiraIssue) {
  var jiraField = allowedValues[0];
  // TODO move this check to the artificial issue container
  return (jiraField.comments.length === jiraField.total) ? jiraField.comments
    : importContext.jiraClient.getComments(jiraIssue.key, importContext.networkFailureHandler).comments;
});

var comments = function (ytFieldName, jiraField, issue, jiraIssue) {
  var createdComments = [];
  var importComment = function (jiraComment) {
    var ytComment = entityConverters.comment(jiraComment, issue, jiraIssue);
    createdComments.push(ytComment);
  };
  // TODO move this check to the artificial issue container
  var jiraComments = (jiraField.comments.length === jiraField.total) ? jiraField.comments
    : importContext.jiraClient.getComments(jiraIssue.key, importContext.networkFailureHandler).comments;
  jiraComments.forEach(importComment);
  return createdComments;
};

var extractWorklogsVisibility = extractAuthorsAndVisibilityGroups('worklog', function (allowedValues, jiraIssue) {
  var jiraField = allowedValues[0];
  // TODO move this check to the artificial issue container
  return (jiraField.worklogs.length === jiraField.total) ? jiraField.worklogs
    : importContext.jiraClient.getWorklogs(jiraIssue.key, importContext.networkFailureHandler).worklogs;
});

var worklogs = function (ytFieldName, jiraField, issue, jiraIssue) {
  var createdWorkItems = [];
  var importWorkItem = function (jiraWorklog) {
    var ytWorkItem = entityConverters.worklog(jiraWorklog, issue, jiraIssue);
    ytWorkItem && createdWorkItems.push(ytWorkItem);
  };
  // TODO move this check to the artificial issue container
  var jiraWorklogs = (jiraField.worklogs.length === jiraField.total) ? jiraField.worklogs
    : importContext.jiraClient.getWorklogs(jiraIssue.key, importContext.networkFailureHandler).worklogs;
  jiraWorklogs.forEach(importWorkItem);
  return createdWorkItems;
};

var extractAttachmentVisibility = extractAuthorsAndVisibilityGroups('attachment', function (allowedValues) {
  return allowedValues;
});

var attachments = function (ytFieldName, jiraField, issue, jiraIssue) {
  var createdAttachments = [];
  jiraField.forEach(function (jiraAttachment) {
    var ytAttachment = entityConverters.attachment(jiraAttachment, issue, jiraIssue);
    ytAttachment && createdAttachments.push(ytAttachment);
  });
  return createdAttachments;
};


var watches = function (jiraIssue, jiraFieldId) {
  if (jiraIssue.fields[jiraFieldId].watchCount == 0) {
    return [];
  }
  var projectKey = jiraIssue.fields.project.key;
  if (!importContext.hasPermission('PROJECT_ADMIN', projectKey) &&
    !importContext.hasPermission('VIEW_VOTERS_AND_WATCHERS', projectKey) &&
    !importContext.hasPermission('MANAGE_WATCHER_LIST', projectKey)) {
    return [];
  }
  var watchers = [];
  importContext.jiraClient.getWatchers(jiraIssue.key, function (wtchs) {
    wtchs.forEach(function (watcher) {
      watchers.push(entityConverters.user(watcher));
    });
    return true;
  }, importContext.networkFailureHandler);
  return watchers;
};

var votes = function (jiraIssue, jiraFieldId) {
  if (jiraIssue.fields[jiraFieldId].votes == 0) {
    return [];
  }
  var projectKey = jiraIssue.fields.project.key;
  if (!importContext.hasPermission('PROJECT_ADMIN', projectKey) &&
    !importContext.hasPermission('VIEW_VOTERS_AND_WATCHERS', projectKey)) {
    return [];
  }
  var voters = [];
  importContext.jiraClient.getVotes(jiraIssue.key, function (vtrs) {
    vtrs.forEach(function (voter) {
      voters.push(entityConverters.user(voter));
    });
    return true;
  }, importContext.networkFailureHandler);
  return voters;
};

var labels = function (modify) {
  return function (ytFieldName, jiraFieldValue, issue, jiraIssue) {
    var lead = issue.project.leader;
    var tags = [];
    jiraFieldValue.forEach(function (label) {
      var tag = entities.IssueTag.getTag(lead, modify(label), true);
      tag.tag(issue);
      tag.shareGroup = entities.UserGroup.allUsersGroup;
      tags.push(tag);
    });
    return tags;
  };
};

var renamer = exports.renamer = function (newName) {
  return function (jiraFieldPrototype) {
    return newName;
  }
};

var dumbConvertName = function (jiraFieldPrototype) {
  return jiraFieldPrototype.id;
};

var ignore_this_field = {
  embedded: true,
  convertSchema: importContext.no_op,
  convertValue: importContext.no_op,
  convertEvent: importContext.no_op,
  transformEvent: importContext.no_op
};


//------------------------------
exports['project'] = ignore_this_field; // it's handled on issue constructor call
exports['id'] = ignore_this_field; // ids can only be auto incremented
exports['issuekey'] = ignore_this_field; // keys can only be auto assigned
exports['lastViewed'] = ignore_this_field; // no such thing in YouTrack
exports['summary'] = {
  embedded: true,
  convertName: dumbConvertName,
  convertSchema: importContext.no_op,
  convertEvent: events.property,
  transformEvent: events.extractFromString()
};
exports['description'] = {
  embedded: true,
  convertName: dumbConvertName,
  convertSchema: importContext.no_op,
  convertEvent: events.property,
  transformEvent: events.extractFromString()
};
exports['created'] = {
  embedded: true,
  convertName: dumbConvertName,
  convertSchema: importContext.no_op,
  convertEvent: importContext.no_op,
  transformEvent: importContext.no_op
};
exports['updated'] = {
  embedded: true,
  convertName: dumbConvertName,
  convertSchema: importContext.no_op
};
exports['resolutiondate'] = ignore_this_field; // retrieve resolution time from resolution events
exports['reporter'] = {
  embedded: true,
  convertName: dumbConvertName,
  convertSchema: function (ytFieldName, project, jiraProject, jiraFieldPrototype, jiraProjectField) {
    if (!jiraProjectField.allowedValues || jiraProjectField.allowedValues.length === 0) {
      jiraProjectField.allowedValues = [null];
    }
    jiraProjectField.allowedValues.forEach(entityConverters.user);
  },
  convertValue: function (ytFieldName, jiraFieldValue, issue) {
    return issue[ytFieldName] = entityConverters.user(jiraFieldValue);
  },
  convertEvent: importContext.no_op // can not change reporter in YouTrack
};
exports['creator'] = ignore_this_field; // no such thing in YouTrack
exports['assignee'] = {
  convertName: renamer('Assignee'),
  convertSchema: function (ytFieldName, project, jiraProject, jiraFieldPrototype, jiraProjectField) {
    var developers = security.developers(project);
    jiraProjectField.allowedValues.map(entityConverters.user).forEach(function (user) {
      user.addToGroup(developers);
    });
  },
  convertEvent: function (event, ytFieldName) {
    event.from = event.from && entityConverters.user({name: event.from});
    event.to = event.to && entityConverters.user({name: event.to});
    events.field(event, ytFieldName);
  }
};
exports['comment'] = {
  embedded: true,
  convertName: renamer('comments'),
  convertSchema: extractCommentsVisibility,
  convertValue: comments,
  convertEvent: function (event, ytFieldName, issue) {
    var fromId = event.change.from;
    var toId = event.change.to;
    if (fromId && !toId) {
      var jiraCommentIds = {};
      event.jiraIssue.fields['comment'].comments.forEach(function (it) {
        jiraCommentIds[it.id] = true;
      });
      var commentToRemove = issue.comments.find(function (it) {
        return it.jiraId && !jiraCommentIds[it.jiraId];
      });
      if (commentToRemove) {
        console.trace('Processing event about deletion of comment ' + commentToRemove.jiraId + ' from issue ' + event.jiraIssue.key);
        commentToRemove.delete();
        event.target = commentToRemove;
        event.from = commentToRemove.deleted;
        event.to = true;
        events.property(event, 'deleted');
      } else {
        console.trace('Could not found comment to delete in issue ' + event.jiraIssue.key);
      }
    }
  },
  transformEvent: importContext.no_op
};
exports['worklog'] = {
  embedded: true,
  convertName: renamer('workitems'), // TODO: implement issue['workitems'] getter
  convertSchema: extractWorklogsVisibility,
  convertValue: worklogs,
  convertEvent: importContext.no_op,
  transformEvent: importContext.no_op
};
exports['attachment'] = {
  embedded: true,
  convertName: renamer('attachments'),
  convertSchema: extractAttachmentVisibility,
  convertValue: attachments,
  convertEvent: function (event, ytFieldName, issue) {
    var fromId = event.change.from;
    var toId = event.change.to;
    if (fromId && !toId) {
      console.trace('Processing event about deletion of attachment ' + fromId);
      var attachmentToRemove = issue.attachments.find(function (it) {
        return it.jiraId === fromId;
      });
      if (attachmentToRemove) {
        issue.removeAttachment(attachmentToRemove);
        console.trace('Deleted attachment ' + fromId + ' from issue ' + event.jiraIssue.key);
        event.target = attachmentToRemove;
        events.removeEntity(event)
      } else {
        console.trace('No attachment ' + fromId + ' found in issue ' + event.jiraIssue.key);
      }
    }
  },
  transformEvent: importContext.no_op // declare as empty to not invoke convertSchema for events
};
exports['watches'] = {
  embedded: true,
  convertSchema: function (ytFieldName, project, jiraProject, jiraFieldPrototype, jiraProjectField, jiraIssue) {
    watches(jiraIssue, 'watches');
  },
  convertValue: function (ytFieldName, jiraField, issue, jiraIssue) {
    watches(jiraIssue, 'watches').forEach(function (user) {
      user.watchIssue(issue);
    });
  }
};
exports['votes'] = {
  embedded: true,
  convertSchema: function (ytFieldName, project, jiraProject, jiraFieldPrototype, jiraProjectField, jiraIssue) {
    votes(jiraIssue, 'votes')
  },
  convertValue: function (ytFieldName, jiraField, issue, jiraIssue) {
    votes(jiraIssue, 'votes').forEach(function (user) {
      issue.voters.add(user);
      issue.votes = issue.votes + 1;
    });
  }
};
exports['labels'] = {
  embedded: true,
  convertName: renamer('tags'),
  convertSchema: importContext.no_op, // not a custom field in YouTrack
  convertValue: labels(function (s) {
    return importUtils.unforbid(s, '<>&,');
  }),
  convertEvent: events.field,
  transformEvent: events.extractFromString(events.splitToArray(' ', function (s) {
    return importUtils.unforbid(s, '<>&,');
  }))
};

exports['issuetype'] = {
  convertType: 'enum[1]',
  convertName: renamer('Type')
};
exports['priority'] = {
  convertType: 'enum[1]'
};
exports['fixVersions'] = {
  convertName: renamer('Fix versions')
};
exports['versions'] = {
  convertName: renamer('Affected versions')
};
exports['status'] = {
  convertName: renamer('State')
};
exports['resolution'] = {
  convertName: renamer('Resolution')
};
exports['components'] = {
  convertName: renamer('Subsystems')
};

exports['thumbnail'] = ignore_this_field; // YouTrack automatically creates thumbnails for all image attachments
exports['project'] = ignore_this_field; // TODO events should be handled with jiraVisibleId property
exports['timespent'] = ignore_this_field; // value equals to timetracking.timeSpentSeconds. Further, YouTrack calculates this value
exports['aggregatetimespent'] = ignore_this_field; // YouTrack calculates this value
exports['timeoriginalestimate'] = ignore_this_field; // value equals to timetracking.timeOriginalEstimateSeconds
exports['aggregatetimeoriginalestimate'] = ignore_this_field; // this includes an estimate over all subtasks
exports['aggregatetimeestimate'] = ignore_this_field; // this includes an estimate over all subtasks
exports['aggregateprogress'] = ignore_this_field; // this includes a progress over all subtasks
exports['timeestimate'] = ignore_this_field; // what's that?
exports['progress'] = ignore_this_field; // quite the same as time spent, or what?
exports['workratio'] = ignore_this_field; // (timeSpent / originalEstimate) * 100
exports['timetracking'] = {
  embedded: true,
  convertSchema: function (ytFieldName, project, jiraProject, jiraPrototype, jiraProjectField) {
    var settings = project.timeTrackingSettings;
    if (!settings) {
      settings = new entities.TimeTrackingSettings(project);
    }
    settings.enabled = true;
    if (!settings.estimate) {
      settings.initDefaults();
    }
    return settings;
  },
  convertValue: function (ytFieldName, jiraFieldValue, issue, jiraIssue) {
    var estimateString = jiraFieldValue['originalEstimate'];
    if (!estimateString) {
      return null;
    }
    var fieldName = issue.project.timeTrackingSettings.estimate.prototype.namePresentation;
    var period = dateTime.toPeriod(jiraFieldValue['originalEstimateSeconds'] * 1000);
    issue[fieldName] = period;
    return period;
  }
};
exports.security = ignore_this_field; // processed inside entityConverters.issue

//---------------------------Issue links----------------------------------

exports['parent'] = {
  embedded: true,
  convertSchema: importContext.no_op,
  convertValue: function (ytFieldName, jiraFieldValue, issue, jiraIssue) {
    console.log("Process parent ");
    perTypeConverters.attachImportLink(jiraIssue, jiraFieldValue, 'Subtask', false)
  },
  transformEvent: events.extractFromId(),
  convertEvent: events.link
};

exports['subtasks'] = {
  embedded: true,
  convertSchema: importContext.no_op,
  convertValue: function (ytFieldName, jiraFieldValue, issue, jiraIssue) {
    jiraFieldValue.forEach(function (subtask) {
      var timestamp;
      var author;
      perTypeConverters.attachImportLink(jiraIssue, subtask, 'Subtask', true);
      // create stub event about linking with subtask, because there are no such events in parent changelog
      var subtaskDetails = importContext.jiraClient.getIssueChangelog(subtask.key, importContext.networkFailureHandler);
      var foundHistory;
      subtaskDetails.changelog.histories.forEach(function (history) {
        history.items.forEach(function (item) {
          if (item.field === 'Parent') {
            foundHistory = history;
          }
        })
      });
      if (foundHistory) {
        author = foundHistory.author;
        timestamp = foundHistory.created;
      } else {
        author = jiraIssue.fields.reporter;
        timestamp = subtaskDetails.fields.created;
      }
      var constructedItem = {
        field: 'subtasks',
        to: subtask.id,
        toString: subtask.key
      };
      events.attachImportEvents(jiraIssue, 'subtasks', issue, entityConverters.user(author), jira.parseTimestamp(timestamp), constructedItem);
    });
  },
  transformEvent: events.extractFromId(),
  convertEvent: events.link
};
exports['issuelinks'] = {
  embedded: true,
  convertSchema: importContext.no_op,
  convertValue: function (ytFieldName, jiraFieldValue, issue, jiraIssue) {
    jiraFieldValue.forEach(function (link) {
      var jiraIssueTo = link.outwardIssue ? link.outwardIssue : link.inwardIssue;
      var typeName = importContext.linkTypes.byJiraId[link.type.id].name;
      var ytLinkTypeStub = constants.defaultLinkTypeMapping[typeName];
      var jiraOutward = link.outwardIssue ? true : false;
      if (ytLinkTypeStub) {
        var ytOutward = ytLinkTypeStub.inversion ? (!jiraOutward) : jiraOutward;
        perTypeConverters.attachImportLink(jiraIssue, jiraIssueTo, typeName, ytOutward);
      } else {
        perTypeConverters.attachImportLink(jiraIssue, jiraIssueTo, typeName, jiraOutward);
      }
    });
  },
  transformEvent: events.extractFromId(function (issueKey) {
    var targetIssue = issueKey && importContext.jiraClient.getIssue(issueKey, importContext.getSkippingFailureHandler([403, 404]));
    return targetIssue && targetIssue.id;
  }),
  convertEvent: events.link
};


//---------------------------Artificial fields----------------------------------
exports['$entity_created'] = {
  embedded: true,
  convertName: importContext.no_op,
  convertEvent: events.addEntity
};
exports['$resolution_timestamp'] = {
  embedded: true,
  convertName: renamer('resolved'),
  convertSchema: importContext.no_op, // not a custom field in YouTrack
  convertEvent: events.property,
  transformEvent: events.extractFromString()
};