/*
 Copyright 2017 JetBrains s.r.o.

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 */

var importUtils = require('@jetbrains/youtrack-scripting-api-import/utils');
var entities = require('@jetbrains/youtrack-scripting-api/entities');
var entityConverters = require('./entityConverters');
var params = require('./importSettings');
var importContext = require('./importContext');

var nameGroup = function (projectName, suffix) {
  return importUtils.unforbid(projectName) + ' ' + suffix;
};

var roleId = function (roles, roleName) {
  return roles[roleName].substr(roles[roleName].lastIndexOf('/') + 1)
};

var developersRole = function () {
  return entities.Role.findOrCreateDeveloper();
};

var adminsRole = function () {
  return entities.Role.findOrCreateProjectAdmin();
};

var developers = exports.developers = function (project) {
  return project.team;
};

var admins = exports.admins = function (project) {
  return entities.UserGroup.findOrCreate(nameGroup(project.name, 'Admins'), project, adminsRole())
};

var visitActors = function (jiraRole, visitor) {
  var actors = jiraRole.actors;
  if (!actors || actors.length === 0) {
    console.warn('Jira role ' + jiraRole.name + ' has no actors');
    return;
  }
  actors.forEach(function (actor) {
    if (actor.type === 'atlassian-group-role-actor') {
      var groupName = actor.name;
      if (!params.useAdminPermission) {
        console.warn('No permissions to retrieve members of group ' + groupName + '. The group is in role of ' + jiraRole.name);
        return;
      }
      var group = tryToRetrieveGroup(groupName);
      if (!group && groupName.indexOf('jira-') === 0) {
        // jira-administrators -> administrators
        group = tryToRetrieveGroup(groupName.substr(6));
        // jira-administrators -> Administrators
        group = group || tryToRetrieveGroup(groupName.charAt(5).toUpperCase() + groupName.substr(6))
      }
      if (group) {
        group.users.items.forEach(visitor);
      } else {
        console.error('Unknown role actor group: ' + actor.name + ' and alternatives names');
      }
    } else if (actor.type === 'atlassian-user-role-actor') {
      visitor(actor);
    } else {
      console.warn('Unknown role actor type: ' + actor.type);
    }
  });
};

var tryToRetrieveGroup = function (name) {
  return importContext.jiraClient.getGroup(name, function (failure) {
      if (failure.code === 404) {
        console.warn('Could not find group ' + name + ', since Jira replied with status code 404');
      } else {
        importContext.networkFailureHandler(failure);
      }
    }
  );
};

var getRoleProcessor = function (project, processAll) {
  return function (jiraRole) {
    var group;
    if (jiraRole.name === 'Developers') {
      group = developers(project);
      project.leader.addToGroup(group); // just in case the leader is not a Developer in this project
      entities.UserRole.grantRoleToGroup(developersRole(), project, group);
    } else if (jiraRole.name === 'Administrators') {
      group = admins(project);
      project.leader.addToGroup(group); // just in case the leader is not an Admin in this project
      entities.UserRole.grantRoleToGroup(adminsRole(), project, group);
    } else if (processAll) {
      group = entities.UserGroup.findOrCreate('jira-role-' + nameGroup(project.name, jiraRole.name));
    }
    if (group) {
      console.log('Group ' + group.name + ' will contain users in role ' + jiraRole.name);
      visitActors(jiraRole, function (jiraUser) {
        entityConverters.user(jiraUser).addToGroup(group);
      });
    }
    return group;
  };
};

exports.processRoles = function (jiraProject, project) {
  var processRole = getRoleProcessor(project);
  if (!jiraProject.$permissions.PROJECT_ADMIN && !params.useAdminPermission) {
    console.warn('No permission to import roles for project ' + jiraProject.key + '. Creating empty project groups.');
    var createRoleStub = function(roleName) {
      processRole({name: roleName, actors: []});
    };
    createRoleStub('Developers');
    createRoleStub('Administrators');
    return;
  }
  console.info('Importing permission-related stuff.');
  var roles = importContext.jiraClient.getRole(jiraProject.key, '', importContext.networkFailureHandler);
  Object.keys(roles).forEach(function (roleName) {
    processRole(importContext.jiraClient.getRole(jiraProject.key, roleId(roles, roleName)));
  });
};

exports.visibilityGroup = function (visibility, jiraProject, project) {
  if (!visibility || !visibility.value) {
    console.warn('Could not create group or role with empty name');
    return null;
  }
  if (visibility.type === 'group') {
    return entityConverters.group({
      name: visibility.value
    });
  } else if (visibility.type === 'role') {
    var roleName = visibility.value;
    var processRole = getRoleProcessor(project, true);
    if (importContext.fieldSchema.projects[jiraProject.key].$permissions.PROJECT_ADMIN || params.useAdminPermission) {
      var projectRoles = importContext.jiraClient.getRole(jiraProject.key, '');
      for (var r in projectRoles) {
        if (r === roleName) {
          return processRole(importContext.jiraClient.getRole(jiraProject.key, roleId(projectRoles, roleName)));
        }
      }
      importUtils.throw('Could not find role with name ' + roleName);
    } else {
      return processRole({name: roleName, actors: []});
    }
  } else {
    importUtils.throw('Unknown visibility type: ' + visibility.type);
  }
};
