var entities = require('@jetbrains/youtrack-scripting-api/entities');
var workflow = require('@jetbrains/youtrack-scripting-api/workflow');

var findResponsibleAndNotify = function(issue, title, body) {
  var user;
  if ((issue.fields.Subsystem || {}).owner) {
    user = issue.fields.Subsystem.owner;
  } else {
    user = issue.project.leader;
  }
  user.notify(title, body);
};

var ONE_HOUR = 1000 * 60 * 60;
var ONE_DAY = ONE_HOUR * 24;

exports.rule = entities.Issue.stateMachine({
  title: workflow.i18n('Define transitions for "State" field with predefined time constraints'),
  fieldName: 'State',
  states: {
    Submitted: {
      initial: true,
      onEnter: function(ctx) {
        ctx.issue.fields.Subsystem = ctx.Subsystem.Support;
      },
      onExit: function(ctx) {
        ctx.issue.fields.required(ctx.Assignee, workflow.i18n('Responsible support engineer is required!'));
      },
      transitions: {
        reproducing: {
          targetState: 'Open'
        },
        incomplete: {
          targetState: 'Incomplete'
        },
        overdue: {
          targetState: 'Overdue',
          after: ONE_HOUR
        }
      }
    },
    Overdue: {
      onEnter: function(ctx) {
        var issue = ctx.issue;
        var user;
        if (issue.fields.Assignee) {
          user = issue.fields.Assignee;
        } else if ((issue.fields.Subsystem || {}).owner) {
          user = issue.fields.Subsystem.owner;
        } else {
          user = issue.project.leader;
        }
        user.notify(workflow.i18n('Acknowledgement needed'), workflow.i18n('Issue {0} is waiting for Acknowledgement.', issue.id));
      },
      transitions: {
        incomplete: {
          targetState: "Incomplete"
        },
        reproducing: {
          targetState: 'Open'
        }
      }
    },
    Open: {
      transitions: {
        wait: {
          after: 4 * ONE_HOUR,
          targetState: 'Wait for reproduce'
        },
        approved: {
          targetState: 'Approved'
        },
        incomplete: {
          targetState: "Incomplete"
        },
        'can\'t reproduce': {
          targetState: 'Can\'t Reproduce'
        }
      }
    },
    'Wait for reproduce': {
      transitions: {
        notify1: {
          after: ONE_DAY,
          targetState: 'Wait for reproduce',
          action: function(ctx) {
            findResponsibleAndNotify(
              ctx.issue, workflow.i18n('Issue is not reproduced in 1 day'), workflow.i18n('Issue {0} is still waiting for reproduction steps.', ctx.issue.id));
          }
        },
        notify4: {
          after: 4 * ONE_DAY,
          targetState: 'Wait for reproduce',
          action: function(ctx) {
            findResponsibleAndNotify(
              ctx.issue,
              workflow.i18n('Issue is not reproduced in 4 days'),
              workflow.i18n('Issue {0} is not reproduced, it\'s better to visit customer on his site.', ctx.issue.id));
          }
        },
        approved: {
          targetState: 'Approved'
        },
        'can\'t reproduce': {
          targetState: 'Can\'t Reproduce'
        },
        incomplete: {
          targetState: 'Incomplete'
        }
      }
    },
    'Can\'t Reproduce': {
      transitions: {
        reopen: {
          targetState: 'Open'
        }
      }
    },
    Incomplete: {
      transitions: {
        reopen: {
          targetState: 'Open'
        }
      }
    },
    Approved: {
      onEnter: function(ctx) {
        ctx.issue.fields.required(ctx.Assignee);
      },
      transitions: {
        fixed: {
          targetState: 'Fixed'
        },
        obsolete: {
          targetState: 'Obsolete'
        }
      }
    },
    Fixed: {
      transitions: {
        verify: {
          targetState: 'Verified'
        },
        reopen: {
          targetState: 'Open'
        }
      }
    },
    Obsolete: {
      transitions: {
        reopen: {
          targetState: 'Open'
        }
      }
    },
    Verified: {
      transitions: {
        reopen: {
          targetState: 'Open'
        }
      }
    }
  },
  requirements: {
    Subsystem: {
      type: entities.OwnedField.fieldType,
      Support: {}
    },
    Assignee: {
      type: entities.User.fieldType
    }
  }
});